///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef _ZLibCompression_CPP
#define _ZLibCompression_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <ShlObj.H>
#include <Stdio.H>
#include <ShlOBJ.H>
#include <Stdlib.H>

#include "ZLibCompression.H"

#include "zConf.h"
#include "zLib.h"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int GZip_Inflate_File(const char *sSourceName, const char *sTargetName)
{
	char *sInBuf = NULL;
	char *sOutBuf = NULL;

	FILE *fSource = NULL;
	FILE *hTarget = NULL;

	int iResult = Z_OK;

	unsigned int iHave = 0;

    z_stream zStream;

	memset(&zStream, 0, sizeof(zStream));

	if((iResult = inflateInit(&zStream)) != Z_OK)
	{
        return iResult;
	}

	if(fopen_s(&fSource, sSourceName, "rb") != 0)
	{
	    deflateEnd(&zStream);
		return 1;
	}

	if(fopen_s(&hTarget, sTargetName, "wb") != 0)
	{
	    deflateEnd(&zStream);
		fclose(fSource);
		return 2;
	}
	
	if((sInBuf = (char *) calloc(sizeof(char), GZIP_CHUNK + 1)) == NULL)
	{
	    deflateEnd(&zStream);
		fclose(fSource);
		fclose(hTarget);
		return 3;
	}

    if((sOutBuf = (char *) calloc(sizeof(char), GZIP_CHUNK + 1)) == NULL)
	{
	    deflateEnd(&zStream);
		free(sInBuf);
		fclose(fSource);
		fclose(hTarget);
		return 4;
	}

    do {
        zStream.avail_in = fread(sInBuf, 1, GZIP_CHUNK, fSource);

        if (zStream.avail_in == 0)
		{
            break;
		}

        zStream.next_in = (Bytef *) sInBuf;

        do {
            zStream.avail_out = GZIP_CHUNK;
            zStream.next_out = (Bytef *) sOutBuf;

            if((iResult = inflate(&zStream, Z_NO_FLUSH)) != Z_OK)
			{
				if(iResult != Z_STREAM_END)
				{
					break;
				}
			}

			iHave = (GZIP_CHUNK - zStream.avail_out);

			if (fwrite(sOutBuf, 1, iHave, hTarget) != iHave || ferror(hTarget))
			{
                iResult = Z_ERRNO;
				break;
            }

			if(iResult != Z_OK)
			{
				break;
			}
        } while (zStream.avail_out == 0);
    } while (iResult != Z_STREAM_END);

    inflateEnd(&zStream);

	free(sInBuf);
    free(sOutBuf);

	fclose(fSource);
	fclose(hTarget);

	return iResult;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int GZip_Deflate_File(const char *sSourceName, const char *sTargetName, int iLevel)
{
	char *sInBuf = NULL;
	char *sOutBuf = NULL;

	FILE *fSource = NULL;
	FILE *hTarget = NULL;

	int iFlush = 0;
	int iResult = Z_OK;

	unsigned int iHave = 0;

	z_stream zStream;

	memset(&zStream, 0, sizeof(zStream));

	if((iResult = deflateInit(&zStream, iLevel)) != Z_OK)
	{
        return iResult;
	}

	if(fopen_s(&fSource, sSourceName, "rb") != 0)
	{
	    deflateEnd(&zStream);
		return 1;
	}

	if(fopen_s(&hTarget, sTargetName, "wb") != 0)
	{
	    deflateEnd(&zStream);
		fclose(fSource);
		return 2;
	}
	
	if((sInBuf = (char *) calloc(sizeof(char), GZIP_CHUNK + 1)) == NULL)
	{
	    deflateEnd(&zStream);
		fclose(fSource);
		fclose(hTarget);
		return 3;
	}

    if((sOutBuf = (char *) calloc(sizeof(char), GZIP_CHUNK + 1)) == NULL)
	{
	    deflateEnd(&zStream);
		free(sInBuf);
		fclose(fSource);
		fclose(hTarget);
		return 4;
	}

    do {
        zStream.avail_in = fread(sInBuf, 1, GZIP_CHUNK, fSource);

		iFlush = feof(fSource) ? Z_FINISH : Z_NO_FLUSH;
        zStream.next_in = (Bytef *)sInBuf;

        do {
            zStream.avail_out = GZIP_CHUNK;
            zStream.next_out = (Bytef *) sOutBuf;

			iResult = deflate(&zStream, iFlush);

			iHave = (GZIP_CHUNK - zStream.avail_out);
            if(fwrite(sOutBuf, 1, iHave, hTarget) != iHave || ferror(hTarget))
			{
                iResult = Z_ERRNO;
				break;
            }
        } while (zStream.avail_out == 0);

		if(iResult != Z_OK)
		{
			break;
		}
	} while (iFlush != Z_FINISH);

    deflateEnd(&zStream);

	free(sInBuf);
    free(sOutBuf);
	
	fclose(fSource);
	fclose(hTarget);

    return iResult;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
